package com.valiantys.nfeed.api;

import java.util.Collection;
import java.util.List;

/**
 * This service gives access to Elements Connect custom field value.
 * Using the methods below, you can set and get custom field content.
 * The value type must match the Elements Connect custom field type :
 * <ul>
 * <li>Elements Connect : java.lang.String (key content)</li>
 * <li>Elements Connect legacy : java.lang.String (key content)</li>
 * <li>Elements Connect Date : java.util.Date</li>
 * <li>Elements Connect DateTime: java.util.Date</li>
 * <li>Elements Connect User : com.atlassian.jira.ApplicationUser</li>
 * </ul>
 */
public interface IFieldValueService {

    /**
     * Sets a Elements Connect custom field value.
     * This works exactly the same as using {@link #setFieldValues(String, String, Collection)} with a Collection containing this single element.
     * Setting null will set an empty collection.
     *
     * @param issueKey      The issue key
     * @param customFieldId the custom field id (customfield_XXX)
     * @param fieldValue    the field value.
     */
    void setFieldValue(String issueKey, String customFieldId, Object fieldValue);

    /**
     * Sets a Elements Connect custom field values.
     * The values must match the type of the custom field (see class declaration javadoc).
     * When setting values on a single value field, only the first value in the collection will be taken into account.
     * If the issue does not exist or if the custom field does not exist or is not an Elements Connect field, the method has no effect.
     *
     * @param issueKey      the issue key
     * @param customFieldId the custom field id
     * @param fieldValues   a collection of values
     */
    void setFieldValues(String issueKey, String customFieldId, Collection<?> fieldValues);

    /**
     * Returns the values contained in the Elements Connect custom field.
     * Single value field return a collection with at most one value.
     * If the issue does not exist or if the custom field does not exist or is not an Elements Connect field, the method returns an empty list.
     *
     * @param issueKey      the issue key
     * @param customFieldId the custom field id
     * @return a collection of values
     */
    Collection<?> getFieldValues(String issueKey, String customFieldId);

    /**
     * Retrieve all custom fields from passed custom fields ids and compute their values according to the given issue.
     * If the issue does not exist or if the custom field does not exist or is not an Elements Connect field, the method has no effect.
     *
     * @param issueKey          the issue key
     * @param customFieldsIds   a list of custom field id
     */
    void computeFieldsValue(String issueKey, List<String> customFieldsIds);

    /**
     * Returns Elements Connect custom field string value as it is stored in Jira database.
     * If the issue does not exist or if the custom field does not exist or is not an Elements Connect field, the method returns null.
     *
     * @param issueKey      the issue key
     * @param customFieldId the custom field id
     * @return the value stored in database
     */
    String getDbValue(String issueKey, String customFieldId);
}
